// lib/feedback/evaluator.ts
import type { Signal, Evaluation } from './types';

export function evaluate(signals: Signal[]): Evaluation {
  if (!signals?.length) {
    return { score: 0, severity: 'NORMAL', confidence: 0.5, reasons: [] };
  }

  const weights: Record<Signal['kind'], number> = {
    FOCUS: -0.05,
    LOAD: 0.25,
    CHURN: 0.3,
    LATENCY: 0.35,
    ERROR: 0.55,
    MEMORY_DRIFT: 0.25,
  };

  let score = 0;
  const reasons: string[] = [];

  for (const s of signals) {
    const w = weights[s.kind] ?? 0.1;
    score += s.value * w;
    if (s.value > 70) reasons.push(`${s.kind} high`);
    else if (s.value > 40) reasons.push(`${s.kind} moderate`);
  }

  score = Math.min(100, Math.max(0, score));
  let severity: Evaluation['severity'] = 'NORMAL';

  if (score >= 65) severity = 'CRITICAL';
  else if (score >= 30) severity = 'WARNING';

  const confidence = Math.min(1, 0.5 + signals.length * 0.05);
  let bucket: Evaluation['bucket'] = 'STABLE';
  if (severity === 'CRITICAL') bucket = 'DEGRADED';
  else if (severity === 'WARNING') bucket = 'VOLATILE';

  return { score, severity, confidence, reasons, bucket };
}
