// lib/evolution/predictive-engine.ts
import type { EvolutionSnapshot } from './timeline';

/**
 * Predict next directive & stability from the last 3 snapshots.
 * - يعطي وزن أعلى لأحدث نقطة (0.2, 0.3, 0.5)
 * - يعزّز تأثير الترند (+/- 0.04) لضمان تجاوزه آخر قيمة عند الاتجاه الصاعد
 * - نفس الـ API المستخدمة في كل أجزاء النظام
 */
/**
 * @file predictive-engine.ts
 * مسؤول عن توليد وتحديث ثقة النظام في التنبؤات القادمة.
 */

let lastConfidence = 0.84;

/**
 * حساب الثقة في آخر تنبؤ (mock logic for testing)
 */
export function getPredictiveConfidence(): number {
  return lastConfidence;
}

/**
 * تحديث الثقة عند كل تنبؤ جديد
 */
export function updatePredictiveConfidence(newValue: number): void {
  lastConfidence = Math.max(0, Math.min(1, newValue)); // clamp بين 0 و 1
}

/**
 * نموذج بسيط لتوليد ثقة بناءً على استقرار النظام
 */
export function calculatePredictiveConfidence(stability: number): number {
  const noise = Math.random() * 0.05;
  const confidence = Math.min(1, Math.max(0, stability * 0.9 + 0.1 + noise));
  updatePredictiveConfidence(confidence);
  return confidence;
}

export function predictNext(snapshots: EvolutionSnapshot[]) {
  if (!Array.isArray(snapshots) || snapshots.length < 3) return null;

  const last = snapshots.slice(-3); // [oldest, middle, latest]

  // Weighted average toward the latest state
  const avgStability = last[0].stability * 0.2 + last[1].stability * 0.3 + last[2].stability * 0.5;

  // Trend direction across the window
  const trend = Math.sign(last[2].stability - last[0].stability) || 0;

  // Map trend -> directive
  const nextDirective = trend > 0 ? 'Expand' : trend < 0 ? 'Stabilize' : 'Hold';

  // Stronger influence of trend (push slightly forward)
  const predicted = +(avgStability + trend * 0.04).toFixed(3);

  // Confidence heuristic
  const confidence = Math.min(1, Math.abs(trend) + 0.7);

  return {
    nextDirective,
    predictedStability: predicted,
    confidence,
  };
}
