"use client";

/**
 * @file hud.glass-effects.tsx
 * @description Unified Glass & Ambient Glow System for all HUD Panels
 * Phase 007.1 – “Glass Sync & Neural Ambient”
 */

import { motion, useAnimationControls } from "framer-motion";
import { useEffect, useMemo } from "react";

/* =============================
   🎨 Core Glass Configuration
============================= */
export const HUDGlass = ({
  status = "Fluctuating",
  children,
  className = "",
}: {
  status?: "Stable" | "Fluctuating" | "Critical";
  children: React.ReactNode;
  className?: string;
}) => {
  const controls = useAnimationControls();

  const glow = useMemo(() => {
    switch (status) {
      case "Stable":
        return { color: "#22c55e", shadow: "0 0 25px rgba(34,197,94,0.35)" };
      case "Fluctuating":
        return { color: "#facc15", shadow: "0 0 25px rgba(250,204,21,0.35)" };
      default:
        return { color: "#ef4444", shadow: "0 0 25px rgba(239,68,68,0.35)" };
    }
  }, [status]);

  useEffect(() => {
    controls.start({
      boxShadow: [
        `0 0 0px ${glow.color}`,
        `0 0 18px ${glow.color}`,
        `0 0 0px ${glow.color}`,
      ],
      transition: { duration: 3.2, repeat: Infinity },
    });
  }, [glow.color]);

  return (
    <motion.div
      animate={controls}
      className={`relative rounded-[24px] p-[2px] overflow-hidden
        bg-gradient-to-r from-cyan-400/30 via-blue-400/20 to-emerald-400/30
        ring-1 ring-cyan-400/10 backdrop-blur-[18px]
        backdrop-saturate-[160%] shadow-[inset_0_0_45px_rgba(0,0,0,0.4)]
        ${className}`}
      style={{
        boxShadow: glow.shadow,
        transition: "box-shadow 0.5s ease-in-out",
      }}
    >
      {/* === Inner Surface Layer === */}
      <div
        className="relative bg-[#010d18]/90 rounded-[22px] p-7
                   shadow-[inset_0_0_40px_rgba(0,160,255,0.18)]
                   text-cyan-50 ring-1 ring-cyan-400/10
                   backdrop-blur-[18px] backdrop-saturate-[180%]"
      >
        {/* === Neon Border Pulse === */}
        <motion.div
          animate={{
            opacity: [0.2, 0.6, 0.2],
            scale: [1, 1.02, 1],
          }}
          transition={{ duration: 3, repeat: Infinity }}
          className="absolute inset-0 rounded-[22px]
                     ring-[1px] ring-cyan-400/20 pointer-events-none"
        />
        {children}
      </div>
    </motion.div>
  );
};

/* =============================
   🌈 Ambient Background Sync
============================= */
export const AmbientBackground = ({
  status = "Fluctuating",
}: {
  status?: "Stable" | "Fluctuating" | "Critical";
}) => {
  const colors = {
    Stable: "radial-gradient(circle at 50% 50%, rgba(34,197,94,0.08), rgba(0,0,0,0.9))",
    Fluctuating:
      "radial-gradient(circle at 50% 50%, rgba(250,204,21,0.08), rgba(0,0,0,0.9))",
    Critical:
      "radial-gradient(circle at 50% 50%, rgba(239,68,68,0.08), rgba(0,0,0,0.9))",
  };

  return (
    <motion.div
      initial={{ opacity: 0 }}
      animate={{ opacity: [0.6, 0.8, 0.6], scale: [1, 1.02, 1] }}
      transition={{ duration: 10, repeat: Infinity }}
      style={{
        background: colors[status],
      }}
      className="absolute inset-0 z-[-1] rounded-[24px] pointer-events-none blur-3xl"
    />
  );
};

/* =============================
   🧠 HUD Wrapper Helper
============================= */
export const HUDContainer = ({
  status = "Fluctuating",
  children,
}: {
  status?: "Stable" | "Fluctuating" | "Critical";
  children: React.ReactNode;
}) => (
  <div className="relative flex flex-col items-center justify-center">
    <AmbientBackground status={status} />
    <HUDGlass status={status}>{children}</HUDGlass>
  </div>
);
