#!/usr/bin/env bash
# ──────────────────────────────────────────────
# 🧠 Jex-HUD CLI v11 — Core Shell
# Author: Jemy-dev  |  Architect of the HUD System
# Description: Universal command-line HUD layer for CI/CD + local ops.
# ──────────────────────────────────────────────

HUD_VERSION="v11.0"
HUD_AUTHOR="Jemy-dev"
HUD_PREFIX="JEX-HUD"
HUD_LOG_DIR=".hud/logs"
HUD_HISTORY_DIR=".hud/history"
mkdir -p "$HUD_LOG_DIR" "$HUD_HISTORY_DIR"

# Colors
C_RESET="\033[0m"
C_CYAN="\033[36m"
C_GREEN="\033[92m"
C_YELLOW="\033[93m"
C_RED="\033[91m"

# ──────────────────────────────────────────────
# 🧱 HUD Banner
hud_banner() {
  local title="${1:-Core CLI}"
  echo -e "\n${C_CYAN}╭────────────────────────────────────────────╮"
  printf "│ %-42s │\n"  "$HUD_PREFIX $HUD_VERSION — $title"
  printf "│ %-42s │\n"  "Architect : $HUD_AUTHOR"
  printf "│ %-42s │\n"  "Runner    : $(hostname)"
  printf "│ %-42s │\n"  "Date      : $(date '+%Y-%m-%d %H:%M:%S')"
  echo -e "╰────────────────────────────────────────────╯${C_RESET}\n"
}

# ──────────────────────────────────────────────
# 📡 Progress Bar
hud_progress() {
  local label="$1"; local total="$2"; local delay="${3:-0.1}"
  echo -ne "$label ["
  for ((i=0; i<total; i++)); do
    echo -ne "${C_GREEN}#${C_RESET}"
    sleep "$delay"
  done
  echo "] ✅"
}

# ──────────────────────────────────────────────
# 🪞 Snapshot
hud_snapshot() {
  local phase="${1:-unknown}"
  local snap_file="$HUD_HISTORY_DIR/snapshot_$(date +%Y%m%d_%H%M%S).log"
  {
    echo "🧠 Jex-HUD Snapshot"
    echo "Phase   : $phase"
    echo "Version : $HUD_VERSION"
    echo "Author  : $HUD_AUTHOR"
    echo "Host    : $(hostname)"
    echo "Git     : $(git rev-parse --short HEAD 2>/dev/null || echo 'no-git')"
    echo "Time    : $(date '+%Y-%m-%d %H:%M:%S')"
  } > "$snap_file"
  echo -e "${C_GREEN}✅ Snapshot stored → ${snap_file}${C_RESET}"
}

# ──────────────────────────────────────────────
# ⚙️ Log Message
hud_log() {
  local type="$1"; shift
  local msg="$*"
  local color="$C_RESET"
  case "$type" in
    info)  color="$C_CYAN";;
    ok)    color="$C_GREEN";;
    warn)  color="$C_YELLOW";;
    err)   color="$C_RED";;
  esac
  echo -e "${color}${HUD_PREFIX}:${type^^}:${C_RESET} $msg" | tee -a "$HUD_LOG_DIR/hud_runtime.log"
}

# ──────────────────────────────────────────────
# 🧩 End Banner
hud_end() {
  echo -e "\n${C_CYAN}╭──────────────────────────────╮"
  echo -e "│ HUD Execution Complete        │"
  printf "│ %s │\n" "$(date '+%Y-%m-%d %H:%M:%S')"
  echo -e "╰──────────────────────────────╯${C_RESET}\n"
}

# ──────────────────────────────────────────────
# Example CLI entry
if [[ "$1" == "demo" ]]; then
  hud_banner "Demo Mode"
  hud_progress "Initializing HUD" 20 0.03
  hud_log info  "Building Jex modules..."
  sleep 0.5
  hud_log ok    "All modules built successfully."
  hud_snapshot  "Demo Phase"
  hud_end
fi
