// tests/feedback/regression.chain.spec.ts
import { describe, it, expect, beforeEach } from 'vitest';
import { runFeedbackCycle } from '../../lib/feedback/runner';
import { getTimeline, clearTimeline } from '../../lib/feedback/memory';
import { getMetrics } from '../../lib/feedback/metrics';
import type { Signal } from '../../lib/feedback/types';

const now = Date.now();

describe('🔗 Regression Chain Test — End-to-End Feedback Flow', () => {
  beforeEach(() => {
    clearTimeline();
  });

  it('✅ should pass through full chain: evaluator → runner → memory → metrics', async () => {
    const signals: Signal[] = [
      { id: 'c1', at: now, kind: 'ERROR', value: 95 },
      { id: 'c2', at: now, kind: 'LATENCY', value: 80 },
      { id: 'c3', at: now, kind: 'LOAD', value: 75 },
    ];

    const result = await runFeedbackCycle(signals, now);
    const timeline = getTimeline();
    const metrics = getMetrics();

    // --- Evaluator check ---
    expect(result.evaln.severity).toBe('CRITICAL');
    expect(result.evaln.score).toBeGreaterThan(70);
    expect(result.evaln.confidence).toBeGreaterThan(0.2);

    // --- Runner check ---
    expect(result.action?.kind).toBe('ROLLBACK');
    expect(result.result?.ok).toBe(true);
    expect(result.blockedBy).toBeNull();

    // --- Memory check ---
    expect(timeline.length).toBe(1);
    expect(timeline[0].evaluation.severity).toBe('CRITICAL');
    expect(timeline[0].chosenAction?.kind).toBe('ROLLBACK');

    // --- Metrics check ---
    expect(metrics.totalCycles).toBeGreaterThan(0);
    expect(metrics.successRate).toBeGreaterThan(0);
  });

  it('🧊 should store multiple cycles and reflect in metrics aggregation', async () => {
    const makeSignals = (v: number): Signal[] => [
      { id: `s${v}-1`, at: now, kind: 'LOAD', value: v },
      { id: `s${v}-2`, at: now, kind: 'LATENCY', value: v },
    ];

    // simulate different cycles (normal → warning → critical)
    await runFeedbackCycle(makeSignals(20), now + 1000);
    await runFeedbackCycle(makeSignals(50), now + 2000);
    await runFeedbackCycle(makeSignals(90), now + 3000);

    const timeline = getTimeline();
    const metrics = getMetrics();

    // timeline sanity
    expect(timeline.length).toBe(3);
    expect(timeline.some((t) => t.evaluation.severity === 'NORMAL')).toBe(true);
    expect(timeline.some((t) => t.evaluation.severity === 'WARNING')).toBe(true);
    expect(timeline.some((t) => t.evaluation.severity === 'CRITICAL')).toBe(true);

    // metrics sanity
    expect(metrics.totalCycles).toBeGreaterThanOrEqual(3);
    expect(metrics.avgLatency).toBeGreaterThan(0);
    expect(metrics.successRate).toBeGreaterThan(0);
  });

  it('🧯 should block low-confidence criticals with safety guard', async () => {
    const signals: Signal[] = [
      { id: 'low1', at: now, kind: 'ERROR', value: 95 },
      { id: 'low2', at: now, kind: 'LATENCY', value: 90 },
      { id: 'low3', at: now, kind: 'LOAD', value: 90 },
    ];

    const result = await runFeedbackCycle(signals, now);
    result.evaln.confidence = 0.1; // simulate degraded trust

    if (result.evaln.confidence < 0.2) {
      expect(result.blockedBy).toBe('safety');
    }
  });
});
