// lib/evolution/timeline.ts

export interface EvolutionSnapshot {
  t: string;
  stability: number;
  adaptationRate: number;
  directive: string;
}

/**
 * يبني الخط الزمني الحالي للنظام من البيانات المخزنة محليًا
 */
export function buildTimeline(): EvolutionSnapshot[] {
  try {
    const raw = localStorage.getItem('EVOLUTION_HISTORY');
    if (!raw) return [];
    const data = JSON.parse(raw);
    if (!Array.isArray(data)) return [];
    return data.slice(-20);
  } catch {
    return [];
  }
}

/**
 * يضيف نقطة جديدة إلى الخط الزمني
 */
export function addSnapshot(snapshot: EvolutionSnapshot) {
  try {
    const key = 'EVOLUTION_HISTORY';
    const prev = JSON.parse(localStorage.getItem(key) || '[]');
    prev.push(snapshot);
    localStorage.setItem(key, JSON.stringify(prev.slice(-30)));
  } catch (err) {
    console.warn('⚠️ [Timeline] Failed to append snapshot', err);
  }
}

/**
 * يحذف الخط الزمني (لأغراض الاختبار)
 */
export function clearTimeline() {
  localStorage.removeItem('EVOLUTION_HISTORY');
}

/**
 * @file timeline.ts
 * مسؤول عن حفظ وتتبع استقرار النظام (stability) عبر الزمن
 * ويُستخدم في التحليل الزمني والـ meta forecast.
 */

export interface TimelineEntry {
  stability: number; // 0 to 1
  timestamp: string;
}

// ذاكرة داخلية مؤقتة (mocked for tests)
let timelineHistory: TimelineEntry[] = [];

/**
 * إضافة نقطة جديدة للتاريخ الزمني
 */
export function addTimelinePoint(stability: number): void {
  timelineHistory.push({
    stability: Math.max(0, Math.min(1, stability)), // clamp
    timestamp: new Date().toISOString(),
  });

  // الحفاظ على آخر 100 نقطة فقط
  if (timelineHistory.length > 100) {
    timelineHistory = timelineHistory.slice(-100);
  }
}

/**
 * استرجاع التاريخ الزمني الحالي
 */
export function getTimelineHistory(): TimelineEntry[] {
  return timelineHistory;
}

/**
 * مسح التاريخ (للاختبارات)
 */
export function clearTimelineHistory(): void {
  timelineHistory = [];
}
