// tests/phase6/neural_summary.spec.ts
import { describe, it, expect } from 'vitest';
import {
  calcNeuralStats,
  generateNeuralSummary,
  toneForConfidence,
  directionForTrend,
} from '@/lib/awareness/collective/analysis/neural-summary-generator';

const mk = (
  t: string,
  conf: number,
  change = 0,
  trend: 'Integrate' | 'Reform' | 'Reset' = 'Reform'
) => ({
  id: Math.random().toString(36).slice(2),
  timestamp: t,
  trend,
  confidence: conf,
  changeRate: change,
  notes: 'test',
});

describe('Neural Summary Generator', () => {
  const logs = [
    mk('2025-10-26T10:00:00Z', 0.45, 1.2, 'Reform'),
    mk('2025-10-26T12:00:00Z', 0.5, 0.0, 'Reform'),
    mk('2025-10-26T14:00:00Z', 0.52, -3.0, 'Reform'),
    mk('2025-10-26T16:00:00Z', 0.5, 2.5, 'Reform'),
  ];

  it('computes stable stats', () => {
    const s = calcNeuralStats(logs as any);
    expect(s.dominantTrend).toBe('Reform');
    expect(s.avgConfidence).toBeGreaterThan(0.49);
    expect(typeof s.momentum).toBe('number');
  });

  it('produces human readable summary', () => {
    const txt = generateNeuralSummary(logs as any);
    expect(txt).toContain('Dominant Trend: Reform');
    expect(txt).toContain('Average Confidence:');
    expect(txt.split('\n').length).toBeGreaterThan(5);
  });

  it('maps tone and direction', () => {
    expect(toneForConfidence(0.8)).toMatch(/high stability/i);
    expect(directionForTrend('Integrate')).toMatch(/cohesion/i);
    expect(directionForTrend('Reset')).toMatch(/disruption/i);
  });
});
