/**
 * useVoiceSynthesis.ts
 * Hook to convert text summaries into speech using Web Speech API
 */

import { useEffect, useRef, useState } from 'react';

export function useVoiceSynthesis() {
  const synthRef = useRef<SpeechSynthesis | null>(null);
  const [isSpeaking, setIsSpeaking] = useState(false);
  const [voice, setVoice] = useState<SpeechSynthesisVoice | null>(null);

  useEffect(() => {
    synthRef.current = window.speechSynthesis;
    const voices = synthRef.current.getVoices();
    const englishVoice = voices.find(
      (v) => v.lang.startsWith('en') && (v.name.includes('Google') || v.name.includes('Microsoft'))
    );
    setVoice(englishVoice || voices[0] || null);
  }, []);

  const speak = (text: string) => {
    if (!synthRef.current || !voice) return;
    synthRef.current.cancel();
    const utter = new SpeechSynthesisUtterance(text);
    utter.voice = voice;
    utter.pitch = 1;
    utter.rate = 1.02;
    utter.volume = 1;
    utter.onstart = () => setIsSpeaking(true);
    utter.onend = () => setIsSpeaking(false);
    synthRef.current.speak(utter);
  };

  const stop = () => {
    synthRef.current?.cancel();
    setIsSpeaking(false);
  };

  return { speak, stop, isSpeaking };
}
