"use client"

import { memo } from "react"
import { motion, AnimatePresence } from "framer-motion"
import { X } from "lucide-react"
import { desktopApps } from "@/data/apps"
import styles from "./ActivitiesOverview.module.css"

interface ActivitiesOverviewProps {
  openWindow: (app: any) => void
  onLogout: () => void
  openWindows: any[]
  focusWindow: (id: string) => void
}

const WindowPreview = memo(function WindowPreview({
  window,
  onClick,
}: {
  window: any
  onClick: () => void
}) {
  return (
    <motion.div
      className={styles.windowPreview}
      onClick={onClick}
      whileHover={{ scale: 1.02 }}
      whileTap={{ scale: 0.98 }}
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.2 }}
    >
      <div className={styles.windowHeader}>
        <span className={styles.windowIcon}>{window.icon}</span>
        <span className={styles.windowTitle}>{window.name}</span>
      </div>
      <div className={styles.windowBody}>
        <div className={styles.windowPlaceholder}>
          <span className={styles.placeholderIcon}>{window.icon}</span>
          <span>{window.name}</span>
        </div>
      </div>
    </motion.div>
  )
})

const AppItem = memo(function AppItem({
  app,
  onClick,
  index,
}: {
  app: any
  onClick: () => void
  index: number
}) {
  return (
    <motion.div
      className={styles.appItem}
      onClick={onClick}
      whileHover={{ scale: 1.05 }}
      whileTap={{ scale: 0.95 }}
      initial={{ opacity: 0, scale: 0.8 }}
      animate={{ opacity: 1, scale: 1 }}
      transition={{ duration: 0.2, delay: index * 0.02 }}
    >
      <span className={styles.appIcon}>{app.icon}</span>
      <span className={styles.appName}>{app.name}</span>
    </motion.div>
  )
})

export default function ActivitiesOverview({
  openWindow,
  onLogout,
  openWindows,
  focusWindow,
}: ActivitiesOverviewProps) {
  const allApps = [
    ...desktopApps,
    { id: "calculator", name: "Calculator", icon: "🧮", type: "app" },
    { id: "notepad", name: "Notepad", icon: "📝", type: "app" },
    { id: "paint", name: "Paint", icon: "🎨", type: "app" },
    { id: "music", name: "Music", icon: "🎵", type: "app" },
    { id: "photos", name: "Photos", icon: "📷", type: "app" },
    { id: "calendar", name: "Calendar", icon: "📅", type: "app" },
  ]

  return (
    <AnimatePresence>
      <motion.div
        className={styles.overlay}
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        exit={{ opacity: 0 }}
        transition={{ duration: 0.15 }}
      >
        <motion.div
          className={styles.activitiesContainer}
          initial={{ scale: 0.95, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          exit={{ scale: 0.95, opacity: 0 }}
          transition={{ duration: 0.2 }}
        >
          <div className={styles.header}>
            <h2>Activities Overview</h2>
            <p>Manage your applications and windows</p>
          </div>

          <div className={styles.content}>
            {/* Running Applications */}
            {openWindows.length > 0 && (
              <div className={styles.section}>
                <h3>Running Applications ({openWindows.length})</h3>
                <div className={styles.windowGrid}>
                  {openWindows.map((window) => (
                    <WindowPreview key={window.id} window={window} onClick={() => focusWindow(window.id)} />
                  ))}
                </div>
              </div>
            )}

            {/* All Applications */}
            <div className={styles.section}>
              <h3>All Applications</h3>
              <div className={styles.appGrid}>
                {allApps.map((app, index) => (
                  <AppItem key={app.id} app={app} onClick={() => openWindow(app)} index={index} />
                ))}
              </div>
            </div>
          </div>

          <div className={styles.footer}>
            <button className={styles.logoutButton} onClick={onLogout} type="button">
              <X size={16} />
              <span>Sign Out</span>
            </button>
          </div>
        </motion.div>
      </motion.div>
    </AnimatePresence>
  )
}
