
# Portfolio OS — Implementation Roadmap (v2: Enhanced After AI Feedback)

> **Owner:** Mohamed Gamal (Jemy)  
> **Maestro (AI Orchestration):** ChatGPT (GPT-5 Thinking)  
> **Assistants:** Gemini CLI, GitHub Copilot  
> **Repo:** `/home/gemmey1020/Documents/JEMY --WORK/tech/me/me/me/me/port`  
> **Source of Truth:** `PORTFOLIO_FULL_ARCHITECTURE_REPORT.md` + AI Feedback v1 (95% alignment)

---

## Table of Contents
1. [Phase 0 — Context & Objectives](#phase-0--context--objectives)
2. [Phase 1 — Critical Layer Fixes](#phase-1--critical-layer-fixes)
3. [Phase 2 — Core Architecture Enhancements](#phase-2--core-architecture-enhancements)
4. [Phase 3 — Feature Enrichment](#phase-3--feature-enrichment)
5. [Phase 4 — Visual & UX Polish](#phase-4--visual--ux-polish)
6. [Phase 5 — Integration, Testing & Automation](#phase-5--integration-testing--automation)
7. [Timeline & Ownership Matrix](#timeline--ownership-matrix)
8. [Runbooks & Commands](#runbooks--commands)
9. [Changelog Policy](#changelog-policy)

---

## Phase 0 — Context & Objectives
**Goal:** Convert the portfolio into a robust **Desktop OS simulation** with clean architecture, excellent UX, and AI-ready documentation.

- **Primary objectives**
  - Centralize window-state + persistence + deep-linking.
  - Data/presentation separation (componentKey + lazy loading).
  - A11y + Reduced motion + Mobile parity.
  - Security (iframe sandbox/CSP).
  - CI + Tests (unit/UI + E2E).
  - Single canonical report for AI (already generated).

**Deliverables**
- Clean codebase with `useWindowManager` managing: open/close/minimize/maximize/focus/z-index/restore.
- Data-driven content (About/Projects/Skills/Qualifications/Resume/Browser).
- `PORTFOLIO_FULL_ARCHITECTURE_REPORT.md` always up to date.
- `IMPLEMENTATION_ROADMAP_PORTFOLIO_OS.md` (this file).

**Definition of Done (Global)**
- No TypeScript errors, ESLint clean.
- Lighthouse OK: LCP < 3s, CLS < 0.1, TBT < 300ms.
- Playwright suite green.
- SEO meta present, manifest present.
- A11y checks (keyboard + focus + reduced motion) pass.

---

## Phase 1 — Critical Layer Fixes
**Why first?** Any feature over a weak base becomes tech debt.

### 1.1 Security: BrowserContent sandbox
- **Task:** Add strict iframe sandbox + allowlist.
- **Files:** `components/windows/BrowserContent.tsx`, `next.config.mjs` (headers).
- **Action:**
  ```tsx
  <iframe
    src={safeUrl}
    sandbox="allow-same-origin allow-scripts allow-popups"
    referrerPolicy="no-referrer"
  />
  ```
  Add headers (content-security-policy) to allow target domains if needed.
- **DOD:** No mixed-content warnings, no console CSP errors.

### 1.2 Error Boundaries
- **Task:** Add a root error boundary + per-window fallback.
- **Files:** `app/error.tsx` or `components/ErrorBoundary.tsx`, wrap Window content with boundary.
- **DOD:** Any runtime exception shows friendly UI, no blank screens.

### 1.3 Baseline Test Harness
- **Task:** Add Vitest + RTL setup files; smoke test for render + one window flow.
- **Files:** `vitest.config.ts`, `tests/smoke/window-manager.spec.tsx`.
- **DOD:** `pnpm test` passes locally.

---

## Phase 2 — Core Architecture Enhancements
**Target:** State/persistence/deep-linking become rock-solid.

### 2.1 Centralize Window State (`useWindowManager`)
- **Task:** Ensure save in `useEffect` (fixed deps) + z-index calculation in `setOpenWindows`.
- **Scope Expansion (per AI feedback):** Must also manage **window positions & sizes**, lifting state out of `Window.tsx`.
- **Files:** `hooks/useWindowManager.ts`.
- **Key snippet (reference):**
  ```ts
  useEffect(() => {
    if (typeof window !== 'undefined') {
      localStorage.setItem(STORAGE_KEY, JSON.stringify({ windows, activeId, nextZ }));
    }
  }, [windows, activeId, nextZ]);
  ```
- **DOD:** Refresh restores same layout; no infinite loops; focusing raises z-index reliably.

### 2.2 Deep-Linking Boot
- **Task:** On mount, parse `?app=`; if exists → open/focus.
- **Files:** `hooks/useWindowManager.ts` (init effect) or `app/page.tsx`.
- **DOD:** Visiting `/?app=about` opens the About window foreground.

### 2.3 Data/Presentation Split
- **Task:** Replace component references in `data/apps.tsx` with `componentKey` strings.
- **Files:** `data/apps.tsx`, `components/Window.tsx`.
- **Window loader:**
  ```tsx
  const Content = useMemo(() => {
    const map: Record<string, () => Promise<any>> = {
      about:    () => import('./windows/AboutContent'),
      projects: () => import('./windows/ProjectsContent'),
      skills:   () => import('./windows/SkillsContent'),
      contact:  () => import('./windows/ContactContent'),
      resume:   () => import('./windows/ResumeContent'),
      browser:  () => import('./windows/BrowserContent'),
      settings: () => import('./windows/SettingsContent'),
      calc:     () => import('./windows/CalculatorContent'),
      music:    () => import('./windows/MusicContent'),
    };
    return lazy(map[app.componentKey] ?? map['about']);
  }, [app.componentKey]);
  ```
- **DOD:** Build size reduced; content loads only when needed.

### 2.4 Migrate Core Components to TypeScript
- **Task:** Convert all remaining `.js` files in `/components` to `.tsx`.
- **Files:** `Desktop.js`, `Window.js`, `Dock.js`, and others.
- **Goal:** Achieve full TypeScript consistency for type safety.
- **DOD:** No `.js` files remain in `/components`; typecheck passes cleanly.

### 2.5 Update Data Models for Future Content
- **Task:** Extend `data/apps.tsx` schema to include new arrays for `qualifications` and `certifications`.
- **Goal:** Prepare structured data model before populating the section in Phase 3.
- **DOD:** Data model extended and exported without runtime errors.

---

## Phase 3 — Feature Enrichment
**Target:** Fill content & usability gaps with data-driven sections.

### 3.1 Qualifications & Certifications
- **Task:** Populate `qualifications` array in `data/apps.tsx`; render in `ResumeContent.tsx` (grouped by year, verify links).
- **DOD:** Real data shown; verify button opens creds; badges for skills/hours.

### 3.2 A11y & Keyboard
- **Task:** ESC clears selection; Ctrl/Cmd+M minimizes active; cycle focus option.
- **Files:** `hooks/useWindowManager.ts`, `Window.tsx`, focus outlines in CSS.
- **DOD:** Fully navigable by keyboard; focus ring visible; respects `prefers-reduced-motion`.

### 3.3 Mobile Parity
- **Task:** `use-mobile` SSR-safe; single tap opens; drag constraints; safe-area padding.
- **DOD:** iPhone/Android usable; no overflow traps.

---

## Phase 4 — Visual & UX Polish
**Target:** Shine and consistency.

### 4.1 Consistent Glassmorphism & Blur
- **Task:** Unify tokens (border, shadow, blur, radii).
- **Files:** CSS modules (Window/Desktop/TopBar/Dock).
- **DOD:** Visual rhythm consistent; dock hover magnification smooth.

### 4.2 Themes (Light/Dark)
- **Task:** Simple theme toggle in `SettingsContent`; CSS vars on `:root`.
- **DOD:** Switch persists (localStorage); colors accessible AA.

### 4.3 Performance Pass
- **Task:** `next/image` for all images; selective `lucide-react` imports; throttle TopBar clock (30–60s).
- **DOD:** Lower main bundle; smoother interactions.

---

## Phase 5 — Integration, Testing & Automation
**Target:** Make it production-grade and AI-operable.

### 5.1 Unit/UI Tests
- **Window manager:** open/close/min/max/focus/order/persistence.
- **Window component:** drag/resize handlers; keyboard shortcuts.
- **DOD:** Coverage on critical logic paths.

### 5.2 E2E (Playwright)
- **Scenarios:** deep-link open → dock focus → minimize/restore → refresh persistence.
- **DOD:** All E2E scenarios pass locally.

### 5.3 CI Pipeline (GitHub Actions)
- **Jobs:** `lint`, `typecheck`, `test`, `build`.
- **DOD:** PRs blocked on red; artifacts uploaded.

### 5.4 AI Orchestration Hooks
- **Gemini CLI:** `scripts/generate-arch-report.sh` (exists).  
- **Policy:** After any merge to `main`, re-run Gemini → update `PORTFOLIO_FULL_ARCHITECTURE_REPORT.md` → commit.

---

## Timeline & Ownership Matrix

| Phase | Tasks (summary) | Owner | Est. |
|------:|------------------|------|------|
| 0 | Context, DOD confirmation | Jemy + Maestro | 0.5d |
| 1 | Security + Boundaries + Test harness | Maestro + Copilot | 1d |
| 2 | State, Persistence, Deep-link, Data split, TSX migration | Maestro + Gemini | 2d |
| 3 | Qualifications, A11y, Mobile parity | Maestro + Copilot | 1.5d |
| 4 | Visual polish, Themes, Perf pass | Copilot (guided) | 1d |
| 5 | Tests (unit/E2E), CI, AI hooks | Maestro + Gemini | 1d |

> **Total:** ~7 days (buffer included).

---

## Runbooks & Commands

### Generate Architecture Report (Gemini CLI)
```bash
cd "/home/gemmey1020/Documents/JEMY --WORK/tech/me/me/me/me/port"
./scripts/generate-arch-report.sh
```

### Test & Build
```bash
pnpm i
pnpm lint
pnpm typecheck
pnpm test
pnpm build
```

### Quick Aliases (optional)
Add to `~/.bashrc`:
```bash
alias gemmy-report='cd "/home/gemmey1020/Documents/JEMY --WORK/tech/me/me/me/me/port" && ./scripts/generate-arch-report.sh'
alias os-run='pnpm dev'
```

---

## Changelog Policy
- Every change that affects architecture/state/UI/data MUST:
  - Update this roadmap (phase/task status).
  - Re-run Gemini CLI and update `PORTFOLIO_FULL_ARCHITECTURE_REPORT.md`.
  - Append a Changelog entry in both files with ISO date.

**Template:**
```markdown
### 2025-10-10
- Phase 2.1: Persistence effect deps fixed.
- Phase 2.4: JS → TSX migration completed.
- Phase 2.5: Data model extended for qualifications.
- Report regenerated via Gemini CLI.
```
