// tests/feedback/regression.spec.ts
import { describe, it, expect } from 'vitest';
import { runFeedbackCycle } from '../../lib/feedback/runner';
import type { Signal } from '../../lib/feedback/types';

const now = Date.now();

describe('🧩 Regression Tests — Phase 5 Final Verification', () => {
  it('✅ should treat very low signals as NORMAL', async () => {
    const signals: Signal[] = [
      { id: 'r1', at: now, kind: 'FOCUS', value: 10 },
      { id: 'r2', at: now, kind: 'LOAD', value: 15 },
    ];
    const result = await runFeedbackCycle(signals, now);
    expect(result.evaln.severity).toBe('NORMAL');
    expect(result.action).toBeNull();
  });

  it('⚠️ should trigger WARNING when moderate load appears', async () => {
    const signals: Signal[] = [
      { id: 'r3', at: now, kind: 'CHURN', value: 50 },
      { id: 'r4', at: now, kind: 'LATENCY', value: 45 },
    ];
    const result = await runFeedbackCycle(signals, now);
    expect(result.evaln.severity).toBe('WARNING');
    expect(result.result?.ok).toBe(true);
  });

  it('🚨 should handle CRITICAL with rollback and record ok:true', async () => {
    const signals: Signal[] = [
      { id: 'r5', at: now, kind: 'ERROR', value: 90 },
      { id: 'r6', at: now, kind: 'LATENCY', value: 80 },
    ];
    const result = await runFeedbackCycle(signals, now);
    expect(result.evaln.severity).toBe('CRITICAL');
    expect(result.result?.ok).toBe(true);
  });

  it('💣 should recover gracefully if execute() fails', async () => {
    const signals: Signal[] = [{ id: 'r7', at: now, kind: 'ERROR', value: 120 }];

    const { execute } = await import('../../lib/feedback/actions');
    const spy = (await import('vitest')).vi
      .spyOn(await import('../../lib/feedback/actions'), 'execute')
      .mockRejectedValueOnce(new Error('Regression simulated fail'));

    const result = await runFeedbackCycle(signals, now);
    expect(result.result?.ok).toBe(false);

    spy.mockRestore();
  });
});
