// tests/feedback/integration.spec.ts
import { describe, it, expect, beforeEach, vi } from 'vitest';
import { runFeedbackCycle } from '../../lib/feedback/index';
import { getTimeline, recordFeedback } from '../../lib/feedback/memory';
import { getMetrics } from '../../lib/feedback/metrics';
import { DefaultPolicy } from '../../lib/feedback/policy';
import type { Signal } from '../../lib/feedback/types';

const now = Date.now();

function mockSignals(type: 'normal' | 'warning' | 'critical'): Signal[] {
  if (type === 'normal') {
    return [
      { id: 's1', at: now, kind: 'FOCUS', value: 20 },
      { id: 's2', at: now, kind: 'LOAD', value: 15 },
    ];
  }
  if (type === 'warning') {
    return [
      { id: 's3', at: now, kind: 'CHURN', value: 50 },
      { id: 's4', at: now, kind: 'LATENCY', value: 45 },
    ];
  }
  return [
    { id: 's5', at: now, kind: 'ERROR', value: 90 },
    { id: 's6', at: now, kind: 'LATENCY', value: 80 },
    { id: 's7', at: now, kind: 'LOAD', value: 70 },
  ];
}

describe('🧠 Feedback & Evaluation Integration Tests', () => {
  beforeEach(() => {
    getTimeline().length = 0;
  });

  it('✅ should handle a normal signal set without triggering any action', async () => {
    const signals = mockSignals('normal');
    const result = await runFeedbackCycle(signals, now);

    expect(result.evaln.severity).toBe('NORMAL');
    expect(result.action).toBeNull();
    expect(getTimeline().length).toBe(1);
  });

  it('⚠️ should detect warning level and adjust rate slightly', async () => {
    const signals = mockSignals('warning');
    const result = await runFeedbackCycle(signals, now);

    expect(result.evaln.severity).toBe('WARNING');
    expect(result.action?.kind).toBe('ADJUST_RATE');
    expect(result.action?.args?.delta).toBeCloseTo(-DefaultPolicy.params.rateStep / 2);
  });

  it('🚨 should trigger rollback when critical error occurs', async () => {
    const signals = mockSignals('critical');
    const result = await runFeedbackCycle(signals, now);

    expect(result.evaln.severity).toBe('CRITICAL');
    expect(result.action?.kind).toBe('ROLLBACK');
    expect(result.result?.ok).toBe(true);
  });

  it('🧊 should respect cooldown and block repeated actions', async () => {
    const signals = mockSignals('critical');

    const first = await runFeedbackCycle(signals, now);
    const second = await runFeedbackCycle(signals, now + 1000); // within cooldown

    expect(first.action?.kind).toBe('ROLLBACK');
    expect(second.blockedBy).toBe('cooldown');
  });

  it('🧯 should prevent action under extremely degraded low-confidence state', async () => {
    const signals: Signal[] = [
      { id: 's8', at: now, kind: 'ERROR', value: 99 },
      { id: 's9', at: now, kind: 'LATENCY', value: 99 },
      { id: 's10', at: now, kind: 'LOAD', value: 95 },
    ];

    const result = await runFeedbackCycle(signals, now);
    const blocked = result.blockedBy === 'safety' || result.blockedBy === 'cooldown';
    expect(blocked).toBe(true);
  });

  it('📈 should correctly update metrics after successful feedback', async () => {
    const signals = mockSignals('critical');
    const result = await runFeedbackCycle(signals, now);

    const metrics = getMetrics();
    expect(metrics.totalCycles).toBeGreaterThan(0);
    expect(metrics.avgLatency).toBeGreaterThanOrEqual(0);
    expect(metrics.successRate).toBeGreaterThanOrEqual(0);
    expect(result.result?.ok).toBe(true);
  });

  it('🧩 should record every feedback cycle in memory timeline', async () => {
    await runFeedbackCycle(mockSignals('normal'), now);
    await runFeedbackCycle(mockSignals('warning'), now + 2000);
    await runFeedbackCycle(mockSignals('critical'), now + 4000);

    const timeline = getTimeline();
    expect(timeline.length).toBe(3);
    expect(timeline[2].evaluation.severity).toBe('CRITICAL');
  });

  it('🔁 should allow recovery from degraded to normal after multiple improvements', async () => {
    await runFeedbackCycle(mockSignals('critical'), now);

    for (let i = 1; i <= 5; i++) {
      await runFeedbackCycle(mockSignals('normal'), now + i * 2000);
    }

    const last = getTimeline().at(-1)!;
    expect(last.evaluation.severity).toBe('NORMAL');
    expect(last.chosenAction).toBeNull();
  });

  it('💣 should handle failed action gracefully without crashing', async () => {
    const badActionSignal: Signal[] = [{ id: 's11', at: now, kind: 'ERROR', value: 120 }];

    const actionsModule = await import('../../lib/feedback/actions');
    const spy = vi
      .spyOn(actionsModule, 'execute')
      .mockRejectedValueOnce(new Error('Simulated failure'));

    const result = await runFeedbackCycle(badActionSignal, now + 8000);
    expect(result.result?.ok).toBe(false);

    spy.mockRestore();
  });

  it('🧮 should generate diverse outcomes across 50 random cycles', async () => {
    const severities = new Set<string>();

    for (let i = 0; i < 50; i++) {
      const signals: Signal[] = Array.from({ length: 3 }).map((_, idx) => ({
        id: `r${i}-${idx}`,
        at: now + i * 6000,
        kind: ['FOCUS', 'CHURN', 'ERROR', 'LATENCY', 'LOAD'][
          Math.floor(Math.random() * 5)
        ] as Signal['kind'],
        value: Math.floor(Math.random() * 100),
      }));

      const result = await runFeedbackCycle(signals, now + i * 6000);
      severities.add(result.evaln.severity);
    }

    expect(severities.size).toBeGreaterThanOrEqual(2);
    expect(getMetrics().totalCycles).toBeGreaterThan(30);
  });
});
