/**
 * تطبيق مخطط إعادة الكتابة على سجل السياسات — مع دعم Dry-Run.
 */
import { MetaConfig } from '../../config/meta.config';
import { loadRegistry, saveRegistry } from '../policy/registry';

type PlanItem = {
  policyId: string;
  valid: boolean;
  action: 'increase_weight_gradually' | 'decrease_weight_or_monitor' | 'none';
  reason: string;
};

export function applyBlueprints(plan: PlanItem[], opts?: { dryRun?: boolean; step?: number }) {
  const dryRun = opts?.dryRun ?? MetaConfig.dryRunByDefault;
  const step = opts?.step ?? 0.05; // 5% خطوة آمنة

  const registry = loadRegistry();
  const updates: Array<{
    policyId: string;
    oldWeight: number;
    newWeight: number;
    dryRun: boolean;
    reason: string;
  }> = [];

  for (const p of plan) {
    if (!p.valid || p.action === 'none') continue;

    const idx = registry.findIndex((r) => r.policyId === p.policyId);
    if (idx === -1) continue;

    const item = registry[idx];
    const oldW = item.weight;
    let newW = oldW;

    if (p.action === 'increase_weight_gradually') {
      newW = clamp(oldW + step, 0.05, 1.0);
      item.trend = '↑';
    } else if (p.action === 'decrease_weight_or_monitor') {
      newW = clamp(oldW - step, 0.05, 1.0);
      item.trend = '↓';
    }

    item.weight = newW;
    item.lastUpdate = new Date().toISOString();

    updates.push({
      policyId: item.policyId,
      oldWeight: oldW,
      newWeight: newW,
      dryRun,
      reason: p.reason,
    });
  }

  if (!dryRun) {
    saveRegistry(registry);
  }

  return { dryRun, step, updates };
}

const clamp = (v: number, min: number, max: number) => Math.max(min, Math.min(max, v));
