"use client";

import { personalInfo, projects, techStack } from "@/data/apps";
import styles from "@/components/windows/WindowContent.module.css";
import { Download } from "lucide-react";

// 🧠 Debug toggle
const DEBUG =
  (typeof process !== "undefined" &&
    process.env.NODE_ENV !== "production") ||
  (typeof process !== "undefined" &&
    process.env.NEXT_PUBLIC_DEBUG === "true");

const dlog = (...args: any[]) => {
  if (DEBUG) console.log("[ResumeContent]", ...args);
};

export default function ResumeContent() {
  const safePersonalInfo = personalInfo || {};
  const safeTechStack =
    techStack && typeof techStack === "object" ? techStack : {};
  const safeProjects = Array.isArray(projects) ? projects : [];

  const handleDownload = () => {
    dlog("⬇️ Resume download clicked");
    alert("Resume download initiated! (This is a demo action)");
    // Real implementation example:
    // window.open('/assets/resume.pdf', '_blank');
  };

  dlog("🧱 Rendering ResumeContent", {
    summary: !!safePersonalInfo.summary,
    techCategories: Object.keys(safeTechStack).length,
    projectsCount: safeProjects.length,
  });

  return (
    <div className={styles.windowContent}>
      {/* ===================== */}
      {/* Professional Summary */}
      {/* ===================== */}
      <section className={styles.contentSection}>
        <h2>Professional Summary</h2>
        {safePersonalInfo.summary ? (
          <p>{safePersonalInfo.summary}</p>
        ) : (
          <p>
            Experienced Full Stack Engineer with a strong focus on scalable
            backend systems, architecture design, and mentoring engineering
            teams.
          </p>
        )}
      </section>

      {/* ===================== */}
      {/* Technical Skills */}
      {/* ===================== */}
      <section className={styles.contentSection}>
        <h2>Technical Skills</h2>

        {Object.keys(safeTechStack).length > 0 ? (
          <div className={styles.skillsGrid}>
            {Object.entries(safeTechStack).map(([category, skills]) => {
              if (!Array.isArray(skills)) return null;
              dlog(`📂 Category: ${category} (${skills.length} skills)`);

              return (
                <div key={category} className={styles.skillCategory}>
                  <h3>
                    {category.charAt(0).toUpperCase() + category.slice(1)}
                  </h3>
                  <ul className={styles.skillList}>
                    {skills.map((skill, index) => (
                      <li key={index} className={styles.skillItem}>
                        {skill.name} ({skill.level || "N/A"})
                      </li>
                    ))}
                  </ul>
                </div>
              );
            })}
          </div>
        ) : (
          <p>No technical skill data available.</p>
        )}
      </section>

      {/* ===================== */}
      {/* Professional Experience */}
      {/* ===================== */}
      <section className={styles.contentSection}>
        <h2>Professional Experience</h2>
        <div className={styles.resumeContainer}>
          {/* Example static items — replace later with dynamic data */}
          <div className={styles.experienceItem}>
            <h4>
              Team Lead & Senior Backend Engineer | Company Name | 2021 – Present
            </h4>
            <p>
              Leading a team of engineers to build robust, secure, and scalable
              systems. Overseeing architecture, DevOps pipelines, and mentoring.
            </p>
            <ul>
              <li>
                Designed RESTful APIs and modular services integrated with
                multiple business units.
              </li>
              <li>
                Improved backend performance by 45% through code optimization
                and Redis caching.
              </li>
              <li>
                Mentored junior developers, conducted code reviews, and led
                agile sprints.
              </li>
            </ul>
          </div>

          <div className={styles.experienceItem}>
            <h4>
              Senior Backend Developer | Another Company | 2018 – 2021
            </h4>
            <p>
              Developed scalable backend applications using Laravel, MySQL, and
              Vue.js for enterprise clients.
            </p>
            <ul>
              <li>
                Collaborated with frontend and QA teams to ensure seamless
                feature integration.
              </li>
              <li>
                Migrated legacy codebases to Laravel 8 and optimized DB queries.
              </li>
              <li>
                Implemented CI/CD using GitHub Actions and Docker containers.
              </li>
            </ul>
          </div>
        </div>
      </section>

      {/* ===================== */}
      {/* Project Portfolio Highlights */}
      {/* ===================== */}
      <section className={styles.contentSection}>
        <h2>Project Portfolio Highlights</h2>

        {safeProjects.length > 0 ? (
          <div className={styles.resumeContainer}>
            {safeProjects.map((project) => {
              const {
                id,
                name,
                role,
                techStack: stack = [],
                description,
                impact,
                link,
              } = project || {};

              dlog(`🚀 Project: ${name || "Untitled"}`);

              return (
                <div key={id || name} className={styles.experienceItem}>
                  <h4>
                    {name || "Untitled Project"}{" "}
                    {role ? `| ${role}` : ""}
                  </h4>
                  {Array.isArray(stack) && stack.length > 0 && (
                    <p>
                      <strong>Tech Stack:</strong> {stack.join(", ")}
                    </p>
                  )}
                  {description && <p>{description}</p>}
                  {impact && (
                    <p>
                      <strong>Impact:</strong> {impact}
                    </p>
                  )}
                  {link && (
                    <a
                      href={link}
                      target="_blank"
                      rel="noopener noreferrer"
                      className={styles.projectLink}
                    >
                      View Project
                    </a>
                  )}
                </div>
              );
            })}
          </div>
        ) : (
          <p>No project data available.</p>
        )}
      </section>

      {/* ===================== */}
      {/* Download Button */}
      {/* ===================== */}
      <button
        onClick={handleDownload}
        className={styles.downloadButton}
        aria-label="Download Resume"
      >
        <Download size={20} style={{ marginRight: "8px" }} />
        Download Full Resume
      </button>
    </div>
  );
}
