"use client"

import { motion } from "framer-motion"
import { useState, useEffect } from "react"
import styles from "./Dock.module.css"
import { desktopApps } from "@/data/apps" // Updated import path

interface DockProps {
  openWindows: any[]
  openWindow: (app: any) => void
  focusWindow: (id: string) => void
  minimizedWindows: string[]
}

export default function Dock({ openWindows, openWindow, focusWindow, minimizedWindows }: DockProps) {
  const [time, setTime] = useState(new Date())

  useEffect(() => {
    const timer = setInterval(() => {
      setTime(new Date())
    }, 1000)
    return () => clearInterval(timer)
  }, [])

  const formatTime = (date: Date) => {
    const hours = date.getHours()
    const minutes = date.getMinutes()
    const ampm = hours >= 12 ? "PM" : "AM"
    const formattedHours = hours % 12 === 0 ? 12 : hours % 12
    const formattedMinutes = minutes < 10 ? "0" + minutes : minutes
    return `${formattedHours}:${formattedMinutes} ${ampm}`
  }

  const dockItemVariants = {
    hidden: { opacity: 0, y: 20 },
    visible: { opacity: 1, y: 0 },
    hover: { scale: 1.1, y: -10, transition: { type: "spring", stiffness: 300 } },
    tap: { scale: 0.9 },
  }

  const dockVariants = {
    hidden: { opacity: 0, y: 50 },
    visible: { opacity: 1, y: 0, transition: { delay: 0.5, duration: 0.5 } },
  }

  const runningApps = openWindows.filter((app) => !minimizedWindows.includes(app.id))

  return (
    <motion.div
      className={styles.dock}
      initial="hidden"
      animate="visible"
      variants={dockVariants}
      aria-label="Application Dock"
    >
      <div className={styles.dockContainer}>
        {desktopApps.map((app) => {
          const isRunning = runningApps.some((runningApp) => runningApp.id === app.id)
          const isMinimized = minimizedWindows.includes(app.id)

          return (
            <motion.div
              key={app.id}
              className={`${styles.dockItem} ${isRunning ? styles.running : ""} ${isMinimized ? styles.minimized : ""}`}
              variants={dockItemVariants}
              whileHover="hover"
              whileTap="tap"
              onClick={() => (isRunning ? focusWindow(app.id) : openWindow(app))}
              aria-label={app.name}
              role="button"
              tabIndex={0}
            >
              <span className={styles.appIcon}>{app.icon}</span>
              {isRunning && <div className={styles.runningDot} />}
            </motion.div>
          )
        })}
      </div>
      <div className={styles.timeDisplay} aria-live="polite">
        {formatTime(time)}
      </div>
    </motion.div>
  )
}
