#!/bin/bash
# ============================================================
# Portfolio OS - AI Architecture Review Script
# Author: Mohamed Gamal (Jemy)
# Maestro: ChatGPT (GPT-5)
# Assistants: Gemini CLI, GitHub Copilot
# ============================================================

# === CONFIG ===
ROOT_PATH="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
DOCS_PATH="$ROOT_PATH/Docs"
LOG_PATH="$ROOT_PATH/logs"
OUTPUT_FILE="$DOCS_PATH/AI_FEEDBACK_PORTFOLIO_OS.md"
MODEL="gemini-2.5-pro"

mkdir -p "$LOG_PATH"

echo "============================================================" | tee -a "$LOG_PATH/review.log"
echo "[AI Review] Starting full architecture analysis..." | tee -a "$LOG_PATH/review.log"
echo "[AI Review] Time: $(date '+%Y-%m-%d %H:%M:%S')" | tee -a "$LOG_PATH/review.log"
echo "[AI Review] Docs Path: $DOCS_PATH" | tee -a "$LOG_PATH/review.log"
echo "[AI Review] Model: $MODEL" | tee -a "$LOG_PATH/review.log"
echo "------------------------------------------------------------" | tee -a "$LOG_PATH/review.log"

# === CHECK DEPENDENCIES ===
if ! command -v gemini &> /dev/null; then
  echo "[ERROR] Gemini CLI not found. Please install it first." | tee -a "$LOG_PATH/review.log"
  exit 1
fi

# === GATHER FILES ===
DOC_FILES=($(find "$DOCS_PATH" -maxdepth 1 -type f -name "*.md" | sort))
if [ ${#DOC_FILES[@]} -eq 0 ]; then
  echo "[ERROR] No markdown files found in $DOCS_PATH" | tee -a "$LOG_PATH/review.log"
  exit 1
fi

echo "[AI Review] Found ${#DOC_FILES[@]} document(s):" | tee -a "$LOG_PATH/review.log"
for file in "${DOC_FILES[@]}"; do
  echo " - $(basename "$file")" | tee -a "$LOG_PATH/review.log"
done
echo "------------------------------------------------------------" | tee -a "$LOG_PATH/review.log"

# === PREPARE PROMPT ===
PROMPT=$(cat <<'EOF'
You are an **AI Architecture Reviewer** for an OS-like portfolio built with Next.js + React + TypeScript.

Analyze the documents provided (architecture report, roadmap, and any related docs).
Produce a detailed feedback report with these sections:
1. **Executive Summary** — brief overview and alignment %
2. **Strengths** — what’s already well-defined
3. **Gaps / Missing Elements** — unimplemented or undefined areas
4. **Contradictions** — mismatches between report and roadmap
5. **Actionable Recommendations** — clear next steps
6. **Quality Metrics** — Confidence %, Code Quality %, Completeness %, Readability %, Security Awareness %

Style: clean Markdown, clear tables, and use emojis for clarity (✅ ⚠️ ❌).
EOF
)

# === CONCAT FILES ===
TMP_INPUT="$LOG_PATH/tmp_review_input.md"
echo "[AI Review] Aggregating documentation files..." | tee -a "$LOG_PATH/review.log"

echo "# Portfolio OS — Combined Review Context" > "$TMP_INPUT"
echo "> Auto-generated at $(date '+%Y-%m-%d %H:%M:%S')" >> "$TMP_INPUT"
echo " " >> "$TMP_INPUT"

for file in "${DOC_FILES[@]}"; do
  echo "---" >> "$TMP_INPUT"
  echo "## File: $(basename "$file")" >> "$TMP_INPUT"
  echo " " >> "$TMP_INPUT"
  cat "$file" >> "$TMP_INPUT"
  echo " " >> "$TMP_INPUT"
done

# === RUN GEMINI REVIEW ===
echo "[AI Review] Running Gemini CLI deep analysis..." | tee -a "$LOG_PATH/review.log"
echo "------------------------------------------------------------" | tee -a "$LOG_PATH/review.log"

gemini -m "$MODEL" -a -p "$PROMPT" < "$TMP_INPUT" > "$OUTPUT_FILE"

EXIT_CODE=$?

if [ $EXIT_CODE -eq 0 ]; then
  echo "[AI Review] ✅ Feedback saved to: $OUTPUT_FILE" | tee -a "$LOG_PATH/review.log"
else
  echo "[AI Review] ❌ Gemini CLI returned an error (exit code $EXIT_CODE)" | tee -a "$LOG_PATH/review.log"
fi

echo "------------------------------------------------------------" | tee -a "$LOG_PATH/review.log"
echo "[AI Review] Done." | tee -a "$LOG_PATH/review.log"
echo "============================================================" | tee -a "$LOG_PATH/review.log"

