// lib/evolution/predictive-engine.ts
import type { EvolutionSnapshot } from './timeline';

/**
 * Predict next directive & stability from the last 3 snapshots.
 * - يعطي وزن أعلى لأحدث نقطة (0.25, 0.35, 0.40)
 * - يعزّز تأثير الترند لتحريك التوقع خطوة صغيرة للأمام (+/- 0.03)
 * - يحافظ على نفس الـ API المستخدمة في الاختبارات وباقي النظام
 */
export function predictNext(snapshots: EvolutionSnapshot[]) {
  if (!Array.isArray(snapshots) || snapshots.length < 3) return null;

  const last = snapshots.slice(-3); // [oldest, middle, latest]

  // Weighted average toward the latest state
  const avgStability =
    last[0].stability * 0.25 + last[1].stability * 0.35 + last[2].stability * 0.4;

  // Trend direction across the window
  const trend = Math.sign(last[2].stability - last[0].stability) || 0;

  // Map trend -> directive
  const nextDirective = trend > 0 ? 'Expand' : trend < 0 ? 'Stabilize' : 'Hold';

  // Stronger influence of trend (push slightly forward)
  const predicted = +(avgStability + trend * 0.03).toFixed(3);

  // Confidence: simple heuristic (0.7 baseline + trend magnitude)
  const confidence = Math.min(1, Math.abs(trend) + 0.7);

  return {
    nextDirective,
    predictedStability: predicted,
    confidence,
  };
}
