// lib/feedback/runner.ts
import { evaluate } from './evaluator';
import { canRun, mark } from './cooldown';
import { execute } from './actions';
import { recordFeedback } from './memory';
import { updateMetrics } from './metrics';
import { DefaultPolicy, chooseAction } from './policy';
import type { Signal, Evaluation } from './types';

/**
 * 🧠 دورة Feedback كاملة ومستقرة
 */
export async function runFeedbackCycle(signals: Signal[], at: number) {
  const evaln: Evaluation = evaluate(signals);
  const policy = DefaultPolicy;

  let action: any = null;
  let result: any = null;
  let blockedBy: string | undefined = undefined;

  try {
    // 🧩 safety guard أولًا وبشكل حتمي
    if (evaln.severity === 'CRITICAL' && evaln.confidence <= 0.25) {
      blockedBy = 'safety';
    }
    // ⏱️ ثم الـ cooldown (إلا في وضع الاختبار)
    const TEST_MODE = process.env.TEST_AWARENESS === 'true';
    if (!blockedBy && !TEST_MODE && !canRun('FEEDBACK', at, policy.cooldownMs)) {
      blockedBy = 'cooldown';
    }

    // ✅ قابل للتنفيذ
    if (!blockedBy) {
      mark('FEEDBACK', at, true);
      action = chooseAction(evaln, policy);

      if (action) {
        try {
          const execResult = await execute(action);
          result = { ...execResult, ok: true };
        } catch (err) {
          console.warn('[runner] Action failed:', (err as Error).message);
          result = { ok: false, error: (err as Error).message || 'Unknown execution error' };
        }
      } else {
        // مفيش فعل فعلي = نجاح افتراضي
        result = { ok: true };
      }
    }

    // 🧩 fallback ثابت لضمان وجود result.ok
    if (!result || typeof result.ok === 'undefined') {
      result = { ok: !blockedBy };
    }

    // 🧾 التسجيل في الذاكرة
    recordFeedback({
      at,
      evaluation: evaln,
      chosenAction: action,
      result,
      policyVersion: policy.version,
      blockedBy,
    });

    // 📊 التحديث بعد التحقق النهائي من result.ok
    updateMetrics(result, at);

    return { evaln, action, result, blockedBy };
  } catch (err) {
    console.error('[runFeedbackCycle] Crash:', err);
    return {
      evaln,
      action,
      result: { ok: false, error: String(err) },
      blockedBy: 'error',
    };
  }
}
