/**
 * Phase 6.x — Integration Runner (Evolution Report)
 * يجمع مخرجات 6.1 → 6.5 ويولّد تقرير شامل عن:
 *   - adaptationRate
 *   - stabilityIndex
 *   - metaCognitionLevel
 *   - nextRecommendedMutation
 * ويكتبها في storage/Evolution_Report.json ويعرض ملخص كونسول أنيق.
 */

import fs from 'node:fs';
import path from 'node:path';

// ====== مصادر البيانات (مسارات التخزين) ======
const STORAGE = {
  ltm: './storage/phase6_long_term_memory.json',
  insights: './storage/conscious_insights.json',
  blueprints: './storage/meta_blueprints.json',
  registry: './storage/policy_registry.json',
  history: './storage/policy_history.json',
  evoLog: './storage/policy_evolution_log.json',
  report: './storage/Evolution_Report.json',
};

// ====== أدوات مساعدة عامة ======
function safeRead<T = any>(p: string, fallback: T): T {
  try {
    if (!fs.existsSync(p)) return fallback;
    const raw = fs.readFileSync(p, 'utf8');
    if (!raw.trim()) return fallback;
    return JSON.parse(raw) as T;
  } catch {
    return fallback;
  }
}

function clamp(v: number, min = 0, max = 1) {
  return Math.max(min, Math.min(max, v));
}

function mean(nums: number[]) {
  return nums.length ? nums.reduce((a, b) => a + b, 0) / nums.length : 0;
}

function pct(n: number) {
  return `${(clamp(n, 0, 1) * 100).toFixed(1)}%`;
}

// ====== تحميل البيانات الخام ======
const LTM = safeRead<{ snapshots?: any[]; metrics?: any }>(STORAGE.ltm, {
  snapshots: [],
  metrics: {},
});
const INSIGHTS = safeRead<any[]>(STORAGE.insights, []);
const BLUEPRINTS = safeRead<any[]>(STORAGE.blueprints, []);
const REGISTRY = safeRead<any[]>(STORAGE.registry, []);
const HISTORY = safeRead<any[]>(STORAGE.history, []);
const EVO_LOG = safeRead<any[]>(STORAGE.evoLog, []);

// ====== 1) adaptationRate ======
// أولوية: ناخدها مباشرة من LTM.metrics.adaptationRate لو موجودة.
// وإلا: نُقدّرها من policy_history (تحسينات موجبة على الأوزان / كل التعديلات).
function computeAdaptationRate() {
  const fromLTM = Number(LTM?.metrics?.adaptationRate);
  if (Number.isFinite(fromLTM)) return clamp(fromLTM, 0, 1);

  // تقدير بديل: نسبة التعديلات الإيجابية
  const deltas = HISTORY.map((h: any) => Number(h.newWeight) - Number(h.oldWeight)).filter(
    Number.isFinite
  );
  if (!deltas.length) return 0;
  const positives = deltas.filter((d) => d > 0).length;
  return clamp(positives / deltas.length, 0, 1) * 0.5; // نرمّيها بنصف الوزن كتحفّظ
}

// ====== 2) stabilityIndex ======
// أولوية: لو عندي LTM.metrics.stability (كل ما كان أصغر كان أفضل) نطبّع إلى [0..1] بالعكس.
// وإلا: نُقدّرها من volatility في INSIGHTS (لو فيها).
function computeStabilityIndex() {
  const rawStab = Number(LTM?.metrics?.stability);
  if (Number.isFinite(rawStab)) {
    // stability (صغير = أفضل). نفترض 0 → 1% نطاق منطقي ونحوّله لعكسي.
    // لو stability ~ 0.0 → index ~ 1.0 (ممتاز) — لو ~0.02 → ~0.8 وهكذا.
    const norm = clamp(1 - clamp(rawStab / 0.02, 0, 1), 0, 1);
    return norm;
  }

  // تقدير بديل: 1 - متوسط volatility من الـ insights (لو موجود)
  const vols = INSIGHTS.map((i: any) => Number(i.volatility)).filter((v) => Number.isFinite(v));
  if (vols.length) {
    const avgVol = mean(vols); // قيم صغيرة = أفضل
    // نفترض 0 → 0.3 مدى معقول للتذبذب
    return clamp(1 - clamp(avgVol / 0.3, 0, 1), 0, 1);
  }

  return 0.5; // محايد لو مفيش بيانات
}

// ====== 3) metaCognitionLevel ======
// مزيج من 3 عناصر: (وجود blueprints) + (قوة الارتباط في insights) + (قبول محاكاة A/B من EVO_LOG)
function computeMetaCognitionLevel() {
  const blueprintSignal = clamp(BLUEPRINTS.length > 0 ? 0.6 : 0.2, 0, 1);

  const corrs = INSIGHTS.map((i: any) => Number(i.correlation))
    .filter((c) => Number.isFinite(c))
    .map((c) => Math.abs(c));
  const avgAbsCorr = corrs.length ? mean(corrs) : 0;
  // نفترض 0..1.25 نطاق معقول للـ |corr|
  const corrSignal = clamp(avgAbsCorr / 1.25, 0, 1);

  const acceptRates = EVO_LOG.map((e: any) => {
    const t = Number(e.tested);
    const a = Number(e.accepted);
    return t > 0 && Number.isFinite(a) ? clamp(a / t, 0, 1) : undefined;
  }).filter((x) => typeof x === 'number') as number[];
  const evoSignal = acceptRates.length ? mean(acceptRates) : 0.3;

  // وزن متوازن
  const level = clamp(0.45 * blueprintSignal + 0.35 * corrSignal + 0.2 * evoSignal, 0, 1);
  return level;
}

// ====== 4) nextRecommendedMutation ======
// القرار النهائي بناءً على:
// - اتجاه majority في BLUEPRINTS (expand/contract)
// - اتجاه registry.trend (↑/↓/→) كترجيح
// - لو stability عالي جدًا والـ adaptationRate منخفض → Freeze
function decideNextMutation(adaptation: number, stability: number) {
  const expands = BLUEPRINTS.filter((b: any) => b.mutation === 'expand').length;
  const contracts = BLUEPRINTS.filter((b: any) => b.mutation === 'contract').length;

  let vote = 'Freeze';
  if (expands > contracts) vote = 'Expand';
  else if (contracts > expands) vote = 'Contract';

  // ترجيح بسيط بالـ trends الحالية:
  const ups = REGISTRY.filter((r: any) => r.trend === '↑').length;
  const downs = REGISTRY.filter((r: any) => r.trend === '↓').length;
  if (ups > downs && vote !== 'Contract') vote = 'Expand';
  if (downs > ups && vote !== 'Expand') vote = 'Contract';

  // قاعدة أمان: لو الاستقرار عالي جدًا (>0.85) لكن التكيّف ضعيف (<0.35) → Freeze
  if (stability > 0.85 && adaptation < 0.35) vote = 'Freeze';

  return vote as 'Expand' | 'Contract' | 'Freeze';
}

// ====== التجميع وإخراج التقرير ======
function buildReport() {
  const adaptationRate = clamp(computeAdaptationRate(), 0, 1);
  const stabilityIndex = clamp(computeStabilityIndex(), 0, 1);
  const metaCognitionLevel = clamp(computeMetaCognitionLevel(), 0, 1);
  const nextRecommendedMutation = decideNextMutation(adaptationRate, stabilityIndex);

  const now = new Date().toISOString();

  const report = {
    timestamp: now,
    inputs: {
      longTermMemory: path.resolve(STORAGE.ltm),
      consciousInsights: path.resolve(STORAGE.insights),
      metaBlueprints: path.resolve(STORAGE.blueprints),
      policyRegistry: path.resolve(STORAGE.registry),
      policyHistory: path.resolve(STORAGE.history),
      evolutionLog: path.resolve(STORAGE.evoLog),
    },
    metrics: {
      adaptationRate: Number(adaptationRate.toFixed(4)),
      stabilityIndex: Number(stabilityIndex.toFixed(4)),
      metaCognitionLevel: Number(metaCognitionLevel.toFixed(4)),
      nextRecommendedMutation,
    },
    registrySnapshot: REGISTRY.map((r: any) => ({
      policyId: r.policyId,
      weight: r.weight,
      trend: r.trend,
      status: r.status,
      lastUpdate: r.lastUpdate,
    })),
    notes: {
      interpretation:
        nextRecommendedMutation === 'Expand'
          ? 'النظام مستقر مع تحسّن إيجابي — التوسيع التدريجي آمن.'
          : nextRecommendedMutation === 'Contract'
          ? 'هناك دلائل على ارتباط سلبي أو تذبذب — يفضّل الانكماش/الخفض.'
          : 'استقرار مرتفع مقابل تكيّف منخفض — التجميد المؤقت مفضل.',
      guidance:
        'استخدم step صغير (3%–5%) مع مراقبة evolutionScore في الدورة القادمة. لا تغيّر أكثر من سياسة واحدة في نفس الدورة إن أمكن.',
    },
  };

  fs.mkdirSync('./storage', { recursive: true });
  fs.writeFileSync(STORAGE.report, JSON.stringify(report, null, 2));

  // ===== Dashboard Console =====
  console.log('\n🧠 Evolution Report — System Dashboard');
  console.table([
    {
      metric: 'adaptationRate',
      value: report.metrics.adaptationRate,
      pretty: pct(report.metrics.adaptationRate),
    },
    {
      metric: 'stabilityIndex',
      value: report.metrics.stabilityIndex,
      pretty: pct(report.metrics.stabilityIndex),
    },
    {
      metric: 'metaCognitionLevel',
      value: report.metrics.metaCognitionLevel,
      pretty: pct(report.metrics.metaCognitionLevel),
    },
    {
      metric: 'nextRecommendedMutation',
      value: report.metrics.nextRecommendedMutation,
      pretty: report.metrics.nextRecommendedMutation,
    },
  ]);

  console.log('\n📦 Saved ->', path.resolve(STORAGE.report));
  console.log('✅ Integration complete.\n');
  return report;
}

// Run
buildReport();
