/**
 * @file meta-forecast.ts
 * تحليل الاتجاه العام للنظام عبر البيانات التاريخية.
 * يعتمد على تحليل معدل التغير، التباين، والزخم (momentum)
 */

import { getTimelineHistory } from './timeline';
import { getPredictiveConfidence } from './predictive-engine';

export interface MetaForecastResult {
  trend: 'Integrate' | 'Reform' | 'Reset';
  confidence: number; // 0 - 1
  changeRate: number; // معدل التغير بين الفترات
  recommendation: string;
  timestamp: string;
}

/**
 * حساب الانحدار البسيط للاتجاه العام (slope)
 */
function calculateSlope(values: number[]): number {
  const n = values.length;
  const meanX = (n - 1) / 2;
  const meanY = values.reduce((a, b) => a + b, 0) / n;
  let num = 0;
  let den = 0;
  for (let i = 0; i < n; i++) {
    num += (i - meanX) * (values[i] - meanY);
    den += Math.pow(i - meanX, 2);
  }
  return den === 0 ? 0 : num / den;
}

/**
 * تحليل الاتجاه العام (Meta Forecast)
 */
export function generateMetaForecast(): MetaForecastResult {
  const history = getTimelineHistory(); // [ { stability: number, timestamp: string }, ... ]
  if (!history.length) {
    return {
      trend: 'Reform',
      confidence: 0.5,
      changeRate: 0,
      recommendation: 'No data available',
      timestamp: new Date().toISOString(),
    };
  }

  const stabilityValues = history.map((h) => h.stability);
  const slope = calculateSlope(stabilityValues);
  //   xxxx
  //   const variance =
  //     stabilityValues.reduce((a, b) => a + Math.pow(b - (a / stabilityValues.length || 0), 2), 0) /
  //     stabilityValues.length;
  //   const lastConfidence = getPredictiveConfidence();

  //   // معدل التغير كنسبة مئوية
  //   const changeRate = slope * 100;

  //   // تحديد الاتجاه العام
  //   let trend: MetaForecastResult['trend'] = 'Reform';
  //   if (slope > 0.02 && variance < 0.015) trend = 'Integrate';
  //   else if (slope < -0.03 || variance > 0.05) trend = 'Reset';

  //   const confidence = Math.min(1, Math.max(0.5, lastConfidence * (1 - variance) + Math.abs(slope)));
  // xxxx

  // === حساب التباين (Variance) بدقة ===
  const mean = stabilityValues.reduce((a, b) => a + b, 0) / stabilityValues.length;
  const variance =
    stabilityValues.reduce((a, b) => a + Math.pow(b - mean, 2), 0) / stabilityValues.length;

  const lastConfidence = getPredictiveConfidence();
  const changeRate = slope * 100;

  // === منطق الاتجاه العام ===
  let trend: MetaForecastResult['trend'] = 'Reform';
  if (slope > 0.015 && variance < 0.03) trend = 'Integrate';
  else if (slope < -0.025 || variance > 0.06) trend = 'Reset';

  // === الثقة النهائية ===
  const confidence = Math.min(
    1,
    Math.max(0.5, lastConfidence * (1 - variance * 0.5) + Math.abs(slope))
  );

  const recommendation =
    trend === 'Integrate'
      ? 'Maintain current trajectory. System stability is improving.'
      : trend === 'Reform'
      ? 'Moderate adjustments recommended. Monitor indicators.'
      : 'Critical instability detected. Initiate reset or rollback.';

  return {
    trend,
    confidence,
    changeRate,
    recommendation,
    timestamp: new Date().toISOString(),
  };
}
