"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { motion } from "framer-motion"
import { format } from "date-fns"
import { Wifi, Volume2, BatteryCharging, ChevronRight } from "lucide-react"
import styles from "./LoginScreen.module.css"

interface LoginScreenProps {
  onUnlock: () => void
}

export default function LoginScreen({ onUnlock }: LoginScreenProps) {
  const [currentTime, setCurrentTime] = useState(new Date())
  const [isTyping, setIsTyping] = useState(false)
  const [password, setPassword] = useState("")

  useEffect(() => {
    const timer = setInterval(() => {
      setCurrentTime(new Date())
    }, 1000)
    return () => clearInterval(timer)
  }, [])

  const handlePasswordChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setPassword(e.target.value)
  }

  const handleLogin = (e: React.FormEvent) => {
    e.preventDefault()
    // For a viewer experience, any input "unlocks" it.
    // In a real app, you'd check credentials here.
    onUnlock()
  }

  const handleInputFocus = () => {
    setIsTyping(true)
  }

  const handleInputBlur = () => {
    if (password === "") {
      setIsTyping(false)
    }
  }

  const backgroundVariants = {
    initial: { opacity: 0 },
    animate: {
      opacity: 1,
      transition: {
        duration: 1,
        ease: "easeOut",
      },
    },
  }

  const cardVariants = {
    hidden: { y: 50, opacity: 0, scale: 0.9 },
    visible: {
      y: 0,
      opacity: 1,
      scale: 1,
      transition: {
        delay: 0.5,
        duration: 0.6,
        ease: "easeOut",
      },
    },
  }

  const particleVariants = {
    animate: {
      y: [0, -1000],
      opacity: [0, 1, 0],
      scale: [0.5, 1.5, 0.5],
      transition: {
        duration: 10 + Math.random() * 10,
        repeat: Number.POSITIVE_INFINITY,
        ease: "linear",
        delay: Math.random() * 5,
      },
    },
  }

  return (
    <motion.div className={styles.loginScreen} initial="initial" animate="animate" variants={backgroundVariants}>
      <div className={styles.gradientBackground} />
      <div className={styles.particlesContainer}>
        {[...Array(50)].map((_, i) => (
          <motion.div
            key={i}
            className={styles.particle}
            variants={particleVariants}
            animate="animate"
            style={{
              left: `${Math.random() * 100}%`,
              top: `${100 + Math.random() * 20}%`, // Start below screen
            }}
          />
        ))}
      </div>

      <div className={styles.statusBar}>
        <div className={styles.statusLeft}>
          <span>{format(currentTime, "HH:mm")}</span>
        </div>
        <div className={styles.statusRight}>
          <Wifi size={16} />
          <Volume2 size={16} />
          <BatteryCharging size={16} />
        </div>
      </div>

      <div className={styles.timeDate}>
        <motion.h1
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.2, duration: 0.5 }}
        >
          {format(currentTime, "h:mm")}
        </motion.h1>
        <motion.p
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.3, duration: 0.5 }}
        >
          {format(currentTime, "EEEE, MMMM d")}
        </motion.p>
      </div>

      <motion.div className={styles.loginCard} initial="hidden" animate="visible" variants={cardVariants}>
        <motion.div
          className={styles.avatar}
          animate={{ rotateY: 360 }}
          transition={{ duration: 10, repeat: Number.POSITIVE_INFINITY, ease: "linear" }}
        >
          <img src="/placeholder.svg?height=100&width=100" alt="User Avatar" />
        </motion.div>
        <h2>Mohamed Gamal</h2>
        <form onSubmit={handleLogin} className={styles.passwordForm}>
          <div className={styles.inputGroup}>
            <input
              type="password"
              placeholder={isTyping ? "" : "Enter to unlock"}
              value={password}
              onChange={handlePasswordChange}
              onFocus={handleInputFocus}
              onBlur={handleInputBlur}
              className={styles.passwordInput}
              aria-label="Password"
            />
            <button type="submit" className={styles.submitButton} aria-label="Unlock">
              <ChevronRight size={20} />
            </button>
          </div>
        </form>
        <p className={styles.hintText}>Click or press Enter to unlock</p>
      </motion.div>
    </motion.div>
  )
}
