import type React from "react";
import type { Metadata } from "next";
import { Inter } from "next/font/google";
import "./globals.css";
import "@/assets/css/os-dashboard.css";

// 🧩 Core Context Providers
import { ReducedMotionProvider } from "@/components/ReducedMotionProvider";
import { DepthProvider } from "@/lib/awareness/collective/components/hud/hud.depth-sync";

// 🧠 HUD Mock Reporter (Phase 9.3)
import { HUDMockReporter } from "@/lib/awareness/collective/components/HUDMockReporter";

const inter = Inter({ subsets: ["latin"] });

export const metadata: Metadata = {
  title: "Mohamed Gamal – Full Stack Engineer",
  description:
    "Team Lead | Full Stack Engineer specializing in Laravel + Vue.js with 7+ years of experience",
  keywords:
    "Mohamed Gamal, Full Stack Engineer, Laravel, Vue.js, Team Lead, Web Developer",
  authors: [{ name: "Mohamed Gamal" }],
  openGraph: {
    title: "Mohamed Gamal – Full Stack Engineer",
    description:
      "Team Lead | Full Stack Engineer specializing in Laravel + Vue.js with 7+ years of experience",
    type: "website",
  },
  generator: "v0.dev",
};

/**
 * 🧩 RootLayout — Phase 9.3 (HUD Neural Mock Aware Layer)
 * --------------------------------------------------------
 * • Wraps the entire OS inside DepthProvider → parallax/glow field.
 * • Adds ReducedMotionProvider for accessibility.
 * • Provides overlay roots for AI/HUD layers.
 * • Integrates HUDMockReporter → visual “Mock Mode Active” HUD.
 */
export default function RootLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  return (
    <html lang="en" suppressHydrationWarning>
      <body className={inter.className}>
        {/* ===============================
             ⚡ Depth-aware Neural Layer
           =============================== */}
        <DepthProvider>
          <ReducedMotionProvider>{children}</ReducedMotionProvider>

          {/* 🧠 Mock HUD Indicator */}
          <HUDMockReporter />

          {/* 🪞 Secondary overlay root (legacy) */}
          <div
            id="global-overlay-root"
            style={{
              position: "fixed",
              inset: 0,
              zIndex: 11990,
              pointerEvents: "none",
            }}
          />

          {/* 🔮 Primary AI Overlay Root */}
          <div
            id="hud-ai-overlay"
            style={{
              position: "fixed",
              inset: 0,
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
              pointerEvents: "none",
              zIndex: 2147483647,
            }}
          />
        </DepthProvider>
      </body>
    </html>
  );
}
 