// lib/feedback/index.ts
import { evaluate } from './evaluator';
import { chooseAction, DefaultPolicy } from './policy';
import { canRun, mark } from './cooldown';
import { guard } from './safety';
import { execute } from './actions';
import { recordFeedback } from './memory';
import { updateMetrics } from './metrics';
import { Signal, Action } from './types';

export async function runFeedbackCycle(signals: Signal[], now = Date.now()) {
  const evaln = evaluate(signals);

  if (!guard(evaln)) {
    await recordFeedback({
      at: now,
      evaluation: evaln,
      chosenAction: null,
      policyVersion: DefaultPolicy.version,
    });
    return { evaln, action: null };
  }

  const action = chooseAction(evaln, DefaultPolicy);
  if (!action) {
    await recordFeedback({
      at: now,
      evaluation: evaln,
      chosenAction: null,
      policyVersion: DefaultPolicy.version,
    });
    return { evaln, action: null };
  }

  if (!canRun(action.kind, now)) {
    await recordFeedback({
      at: now,
      evaluation: evaln,
      chosenAction: action as Action,
      policyVersion: DefaultPolicy.version,
      blockedBy: 'cooldown',
    });
    return { evaln, action, blockedBy: 'cooldown' };
  }

  const result = await execute(action as Action);
  mark(action.kind, now, result.ok);
  updateMetrics(result.ok, result.latencyMs);

  await recordFeedback({
    at: now,
    evaluation: evaln,
    chosenAction: action as Action,
    result,
    policyVersion: DefaultPolicy.version,
  });

  return { evaln, action, result };
}
